/*
  pathabbr.c v2005-02-10
  Abbreviate paths.

  Copyright (C) 2002 David Necas (Yeti) <yeti@physics.muni.cz>.
  Copyright (C) 2005 Rizwan Kassim <rizwank@geekymedia.com>

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the Free
  Software Foundation; either version 2 of the License, or (at your option)
  any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
*/
#define _GNU_SOURCE
#include <stdlib.h>
#include <math.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <assert.h>
#include <stdbool.h>
#include <stdarg.h>
#include <getopt.h>
#include <regex.h>
/*// why doesnt  #ifdef CYGWIN work here?*/
#include "getline.c"
#include "stpncpy.c"

#define SPACER "                         "

#ifdef __GNUC__
# define PVAR(f, v) fprintf(stderr, __FILE__ ":%u " __FUNCTION__ "(): " \
                            #v " == %" #f "\n", __LINE__, v)
#else /* __GNUC__ */
# define PVAR(f, v) fprintf(stderr, __FILE__ ":%u " \
                            #v " == %" #f "\n", __LINE__, v)
#endif /* __GNUC__ */


static const size_t LIMIT_MAX_LEN = 16384;

static const size_t DEFAULT_MAX_LEN = 25;
static const char *DEFAULT_ELLIPSIS = "..";
static const double DEFAULT_PREFERRED_RATIO = 0.5;

static const char *DEFAULT_RATING_RULES[] = {
  "30::/",
  "20::$",
  "20::^",
  "10::[-._ ]"
};

char *program_invocation_short_name = "pathabbr";
  
typedef struct RatingRule {
  char *pattern;
  regex_t *compiled;
  double ledge, redge;
  struct RatingRule *next;
} RatingRule;

/* function prototypes {{{1 */
static char * abbreviate(char *string, RatingRule *rr,
                         const size_t max, char *result);
static double * compute_ratings(char *string, RatingRule *rr);
static void compile_regexps(RatingRule *rr);
static void process_options(int argc, char *argv[]);
static RatingRule * add_rating_rule(RatingRule *rr, const char *rule);
static void print_help(void);
static void print_version(void);
static void pathabbr_error(const bool fatal, const char *const message, ...);

/* global variables {{{1 */
size_t max_len;
char *ellipsis;
size_t el;
double preferred_ratio;
RatingRule *rating_rules;

/* inline functions {{{1 */
static inline void*
xmalloc(size_t size)
{
  void *p = malloc(size);
  assert(p != NULL);
  return p;
}

static inline void *
xrealloc(void *ptr, size_t size)
{
  void *p = realloc(ptr, size);
  assert(p != NULL);
  return p;
}

/* main(int, char**) {{{1 */
int main(int argc, char *argv[])
{
  size_t len = 0;
  ssize_t nl_pos;
  char *line = NULL;
  char *abbr = NULL;

  process_options(argc, argv);
  while ((nl_pos = getline(&line, &len, stdin)) > 0) {
    line[nl_pos - 1] = '\0'; /* get rid of \n */
    abbr = abbreviate(line, rating_rules, max_len, abbr);
    puts(abbr);
  }

  return 0;
}

/* abbreviate(char*, RatingRule*, const size_t, char*) {{{1 */
static char *
abbreviate(char *string, RatingRule *rr, const size_t max, char *result)
{
  ssize_t pos, rpos, best_pos, len;
  double *ratings;
  double q, q_min;
  char *p;

  assert(max > el);
  if (result == NULL) result = (char*)malloc(max + 1);

  len = strlen(string);
  if (len <= max) return strcpy(result, string);

  if (max == el) return strcpy(result, ellipsis);

  ratings = compute_ratings(string, rr);
  best_pos = 0;
  q_min = HUGE_VAL;
  for (pos = 0; pos <= max - el; pos++) {
    rpos = max - pos - el;
    q = (double)(rpos - pos)/(max - el) - preferred_ratio;
    q = 100*q*q;
    q -= hypot(ratings[pos], ratings[len - rpos]);
    if (q <= q_min) {
      q_min = q;
      best_pos = pos;
    }
  }
  free(ratings);

  rpos = max - best_pos - el;
  p = stpncpy(result, string, best_pos);
  p = stpncpy(p, ellipsis, el);
  p = stpncpy(p, string + len - rpos, rpos);
  result[max] = '\0';

  return result;
}

/* compute_ratings(char*, RatingRule*) {{{1 */
static double *
compute_ratings(char *string, RatingRule *rr)
{
  int err;
  int flags;
  size_t offset, len;
  regmatch_t match;
  double *ratings;

  assert(string);
  len = strlen(string);
  ratings = (double*)xmalloc((len + 1)*sizeof(double));
  for (offset = 0; offset <= len; offset++) ratings[offset] = 0;

  for (; rr; rr = rr->next) {
    flags = 0;
    offset = 0;
    do {
      err = regexec(rr->compiled, string + offset, 1, &match, flags);
      if (err == REG_NOMATCH) break;
      assert(err == 0);
      if (ratings[offset + match.rm_so] < rr->ledge)
        ratings[offset + match.rm_so] = rr->ledge;
      if (ratings[offset + match.rm_eo] < rr->redge)
        ratings[offset + match.rm_eo] = rr->redge;
      offset += match.rm_eo ? match.rm_eo : 1;
      flags = REG_NOTBOL;
    } while (offset <= len);
  }

#ifdef DEBUG
  {
    size_t i;

    for (i = 0; i <= len; i++)
      fprintf(stderr, "%c_%0.0f ", string[i], ratings[i]);

    fputc('\n', stderr);
  }
#endif

  return ratings;
}

/* compile_regexps(RatingRule*) {{{1 */
static void
compile_regexps(RatingRule *rr)
{
  int err;

  for (; rr; rr = rr->next) {
    rr->compiled = (regex_t*)xmalloc(sizeof(regex_t));
    err = regcomp(rr->compiled, rr->pattern, REG_EXTENDED);
    if (err) {
      size_t msg_size = regerror(err, rr->compiled, NULL, 0);
      char *errmsg = (char*)xmalloc(msg_size);
      regerror(err, rr->compiled, errmsg, msg_size);
      pathabbr_error(true, "Invalid regex `%s': %s", rr->pattern, errmsg);
    }
  }
}

/* process_options(int, char**) {{{1 */
static void
process_options(int argc, char *argv[])
{
  static const char *short_options = "d:e:hr:v";

  static const struct option long_options[] = {
    { "delimiter-rule", required_argument, NULL, 'd' },
    { "ellipsis", required_argument, NULL, 'e' },
    { "help", no_argument, NULL, 'h' },
    { "preferred-ratio", required_argument, NULL, 'r' },
    { "version", no_argument, NULL, 'v' },
    { NULL, 0, NULL, 0 }
  };

  int c;

  max_len = DEFAULT_MAX_LEN;
  preferred_ratio = DEFAULT_PREFERRED_RATIO;
  rating_rules = NULL; /* use default only if user didn't specify any rules */
  ellipsis = strdup(DEFAULT_ELLIPSIS);
  el = strlen(ellipsis);

  while ((c = getopt_long(argc, argv, short_options,
                          long_options, NULL)) != -1) {
    switch (c) {
      case '?': /* unknown option */
      case ':': /* missing paramter */
      exit(EXIT_FAILURE);
      break;

      case 'h': /* help (and exit) */
      print_help();
      exit(EXIT_SUCCESS);
      break;

      case 'v': /* version (and exit) */
      print_version();
      exit(EXIT_SUCCESS);
      break;

      case 'e': /* ellipsis */
      free(ellipsis);
      ellipsis = strdup(optarg);
      el = strlen(ellipsis);
      break;

      case 'r': /* second/first part ratio */
      if (!sscanf(optarg, "%lf", &preferred_ratio))
        pathabbr_error(true, "Invalid value of preferred ratio `%s'",
                       optarg);
      break;

      case 'd': /* delimiter regex */
      rating_rules = add_rating_rule(rating_rules, optarg);
      break;

      default:
      abort();
      break;
    }
  }

  /* max length */
  if (optind + 1 < argc) pathabbr_error(true, "Too many non-option arguments");
  if (optind + 1 == argc) {
    if (!sscanf(argv[optind], "%u", &max_len)
        || max_len < el
        || max_len > LIMIT_MAX_LEN)
      pathabbr_error(true, "Invalid value of maximal length `%s'",
                     argv[optind]);
  }

  /* rating rules */
  if (!rating_rules) {
    int i;

    for (i = 0; i < sizeof(DEFAULT_RATING_RULES)/sizeof(char*); i++)
      rating_rules = add_rating_rule(rating_rules, DEFAULT_RATING_RULES[i]);
  }

  compile_regexps(rating_rules);
}

/* add_rating_rule(RatingRule*, char*) {{{1 */
static RatingRule *
add_rating_rule(RatingRule *rr, const char *rule)
{
  RatingRule *rnew;
  const char *p, *rule_orig;

  rule_orig = rule;
  rnew = (RatingRule*)xmalloc(sizeof(RatingRule));
  p = strchr(rule, ':');
  if (!p) goto BAD_RULE;
  if (!sscanf(rule, "%lf:", &(rnew->ledge))
      || rnew->ledge <= 0) goto BAD_RULE;
  rule = p+1;
  p = strchr(rule, ':');
  if (!p) goto BAD_RULE;
  if (p == rule) rnew->redge = rnew->ledge;
  else {
    if (!sscanf(rule, "%lf:", &(rnew->redge))
        || rnew->redge <= 0) goto BAD_RULE;
  }
  rule = p+1;
  if (strlen(rule) == 0) goto BAD_RULE;
  rnew->pattern = strdup(rule);
  rnew->compiled = NULL;
  rnew->next = rr;
  return rnew;

  BAD_RULE:
  pathabbr_error(true, "Invalid delimiter rule `%s'", rule_orig);
  return NULL; /* cannot be reached */
}

/* print_help() {{{1 */
static void
print_help(void)
{
  int i;

  printf("Usage:  %s [OPTION]... MAX-LENGTH <PATHLIST\n",
         program_invocation_short_name);
  puts("Abbreviates paths to given length using ellipsis.");
  puts("Options:");
  printf(" -d, --delimiter-rule=LEFT:RIGHT:DELIMITER-REGEXP\n"
         SPACER "add a delimiter rule with left edge bonus LEFT, right\n"
         SPACER "edge bonus RIGHT (may be omitted if equal to LEFT)\n"
         SPACER "and regular expression DELIMITER-REGEXP\n"
         SPACER "(default:");
  for (i = 0; i < sizeof(DEFAULT_RATING_RULES)/sizeof(char*); i++) {
    printf(" `%s'", DEFAULT_RATING_RULES[i]);
    printf(i == sizeof(DEFAULT_RATING_RULES)/sizeof(char*) - 1 ? ")\n" : ",");
  }
  printf(" -e, --ellipsis=STRING   "
         "string used as ellipsis (default: `%s')\n",
         DEFAULT_ELLIPSIS);
  printf(" -r, --preferred-ratio=FLOAT\n"
         SPACER "preferred path part length relative difference\n"
         SPACER "(in range -1.0..1.0, default: %0.1f)\n", DEFAULT_PREFERRED_RATIO);
  puts(" -h, --help              print this help and terminate");
  puts(" -v, --version           print version info and terminate");
  puts("");
  puts("Path list is read from standard input and written to standard output.");
  puts("Default rule set is used only when no rules are specified.");
  puts("");
  puts("Report bugs to <yeti@physics.muni.cz>\n");
  puts("Cygwin compile errors should be reported to <rizwank@geekymedia.com>\n (please include `"
       PACKAGE "' in subject)");
}

/* print_version() {{{1 */
static void
print_version(void)
{
  puts(PACKAGE " " VERSION "\n");
  puts("Copyright (C) 2002 David Necas (Yeti) <yeti@physics.muni.cz>\n");
  puts("Copyright (C) 2005 Rizwan Kassim <rizwank@geekymedia.com>\n");
  puts(
"This is free software; you can be copy and/or modify it under the terms\n"
"of the GNU General Public License as published by the Free Software\n"
"Foundation; either version 2 of the License, or (at you option) any\n"
"later version.  Please see COPYING for details.  There is NO WARRANTY;\n"
"not even for MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE."
  );
}

/* pathabbr_error(const bool, const char*const, ...) {{{1 */
static void
pathabbr_error(const bool fatal, const char *const message, ...)
{
  va_list ap;
  int err;

  err = errno;
  fprintf(stderr, "%s: ", program_invocation_short_name);
  va_start(ap, message);
  vfprintf(stderr, message, ap);
  va_end(ap);
  if (err) fprintf(stderr, ": %s", strerror(err));
  fputc('\n', stderr);
  if (fatal) exit(EXIT_FAILURE);
}
